/*
 * Copyright (C) 2008 The Android Open Source Project
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef _MOT_PTHREAD_H_
#define _MOT_PTHREAD_H_

#include <time.h>
#include <signal.h>
#include <sched.h>
#include <limits.h>
#include <sys/types.h>

/*
 * Types
 */
typedef struct
{
    int volatile value;
} mot_pthread_mutex_t;


#define  MOT_PTHREAD_MUTEX_INITIALIZER             {0}
#define  MOT_PTHREAD_RECURSIVE_MUTEX_INITIALIZER   {0x4000}
#define  MOT_PTHREAD_ERRORCHECK_MUTEX_INITIALIZER  {0x8000}

enum {
    MOT_PTHREAD_MUTEX_NORMAL = 0,
    MOT_PTHREAD_MUTEX_RECURSIVE = 1,
    MOT_PTHREAD_MUTEX_ERRORCHECK = 2,

    MOT_PTHREAD_MUTEX_ERRORCHECK_NP = MOT_PTHREAD_MUTEX_ERRORCHECK,
    MOT_PTHREAD_MUTEX_RECURSIVE_NP  = MOT_PTHREAD_MUTEX_RECURSIVE,

    MOT_PTHREAD_MUTEX_DEFAULT = MOT_PTHREAD_MUTEX_NORMAL
};


typedef struct
{
    int volatile value;
} mot_pthread_cond_t;


/*
 * Prototypes
 */
#if __cplusplus
extern "C" {
#endif

int mot_pthread_mutex_lock(mot_pthread_mutex_t *mutex);
int mot_pthread_mutex_unlock(mot_pthread_mutex_t *mutex);
int mot_pthread_mutex_trylock(mot_pthread_mutex_t *mutex);

int mot_pthread_cond_broadcast(mot_pthread_cond_t *cond);
int mot_pthread_cond_signal(mot_pthread_cond_t *cond);
int mot_pthread_cond_wait(mot_pthread_cond_t *cond, mot_pthread_mutex_t *mutex);
int mot_pthread_cond_timedwait(mot_pthread_cond_t *cond,
                           mot_pthread_mutex_t * mutex,
                           const struct timespec *abstime);

/* BIONIC: same as pthread_cond_timedwait, except the 'reltime' given refers
 *         is relative to the current time.
 */
int mot_pthread_cond_timedwait_relative_np(mot_pthread_cond_t         *cond,
                                     mot_pthread_mutex_t        *mutex,
                                     const struct timespec  *reltime);

//#define HAVE_PTHREAD_COND_TIMEDWAIT_RELATIVE 1


#if __cplusplus
} /* extern "C" */
#endif

#endif // _MOT_PTHREAD_H_
